using System;
using System.IO;
using System.Net;
using System.Xml;
using System.Threading;
using System.Diagnostics;
using System.Collections;
using System.Net.Sockets;

using Team_Project;
using Team_Project.Elements;
using Team_Project.Exceptions;
using Team_Project.PersistencyManagers.Storages;
using Team_Project.PersistencyManagers.Protocols;

namespace Team_Project
{
	/// <summary>
	/// Questa classe  un Helper per l'attesa di messaggi TCP da parte delle
	/// altre copie.
	/// </summary>
	/// <remarks>
	/// <para>Nella sua implementazione attuale (prototipale) non  possibile
	/// registrare pi di un handler per tipo di messaggio.</para>
	/// <para>I messaggi interpretati da questa classe devono necessariamente
	/// essere cos composti: <code>&lt;TipoMessaggio&gt;\r\n&lt;Corpo&gt;</code></para>
	/// <para>L'oggetto globale di questa copia  disponibile con il nome
	/// "MessageListener"</para>
	/// </remarks>
	[TeamProjectInit(true,"MessageListener")]
	public class MessageListener :MarshalByRefObject, IDisposable
	{
		#region Required by TeamProjectInit
		/// <summary>
		/// Inizializza un'istanza di MessageListener
		/// </summary>
		/// <remarks>Questa funzione  richiesta per l'inizializzazione
		/// effettuata dall'Engine</remarks>
		/// <returns>Un message listener configurato.</returns>
		public static MessageListener ComponentInit()
		{
			return new MessageListener();
		}
		#endregion

		/// <summary>
		/// Porta di ascoloto per i messaggi (TCP) gestiti da questa classe.
		/// </summary>
		public static int listenPort = -1;
		/// <summary>
		/// Thread di ascolto dei messaggi
		/// </summary>
		protected Thread t;
		/// <summary>
		/// Tempo (in milli-secondi) ogni quano viene controllato se  arrivato
		/// un nuovo messaggio
		/// </summary>
		public int PollingTime = 20;
		/// <summary>
		/// Hahtable contenente i riferimenti agli handler dei messaggi
		/// </summary>
		[NonSerialized]
		protected Hashtable registred = new Hashtable();
		
		/// <summary>
		/// Costruisce un'istanza di listener. L'ascolto dei messaggi inizia
		/// immediatamente (anche se non c' alcun handler registrato) su un
		/// altro thread.
		/// </summary>
		public MessageListener()
		{
			t = new Thread(new ThreadStart(ListenerStart));
			t.Start();
		}

		/// <summary>
		/// Ottiene la porta sulla quale il message listener  in attesa
		/// </summary>
		public int ListenPort
		{
			get
			{
				if(listenPort <= 0)
				{
					listenPort = MyDNS.ListenerPort(Globals.Instance.LocalCopyName);
				}
				return listenPort;
			}
		}

		/// <summary>
		/// Registra un tipo per la gestione di un tipo di messaggio.
		/// </summary>
		/// <remarks><para>I messaggi vengono identificati dalla parte di messaggio
		/// fino al primo <i>new line</i>, il contenuto successivo pu essere
		/// codificato in qualsiasi formato e sar riconosciuto dal'handelr</para>
		/// <para>Un solo listener pu essere registrato per ogni tipo di messaggio
		/// in quanto non  implementato un mecanismo di multicast verso gli oggetti.</para>
		/// </remarks>
		/// <exception cref="System.Exception">Se esiste gi un listener registrato
		/// per il tipo di messaggio.</exception>
		/// <exception cref="System.ArgumentException">Se il tipo passato come parametro (handler)
		/// non implementa l'interfaccia <see cref="Team_Project.IMessageHandler">IMessageHandler</see></exception>
		/// <param name="message">Header che identifica la tipologia di messaggio</param>
		/// <param name="handler">Tipo che verr istanziato per la gestione del messaggio
		/// Il tipo deve derivare da <see cref="Team_Project.IMessageHandler">IMessageHandler</see></param>
		public void RegisterListener(string message, Type handler)
		{
			object hl = registred[message];
			if(hl == null)
			{
				if(!typeof(IMessageHandler).IsAssignableFrom(handler))
					throw new ArgumentException("Handler type does not support IMessageHandler","handler");
				registred.Add(message,handler);
				Trace.WriteLine(handler.Name + " registred for message " + message);
			}
#if DEBUG
			else if(!hl.Equals(handler))
				throw new Exception("An handler is already registred for message "+ message);
#endif
		}

		/// <summary>
		/// Deregistra un listener per un messaggio.
		/// </summary>
		/// <exception cref="System.Exception">Nel caso il tipo non sia
		/// registrato per il messaggio</exception>
		/// <param name="message">Header del messaggio per il quale si
		/// desidera deregistrare l'handler</param>
		/// <param name="handler">Tipo dell'handler da deregistrare</param>
		public void UnregisterListener(string message, Type handler)
		{
#if DEBUG
			if(!registred.ContainsKey(message))
				throw new Exception("No handlers for message " + message);
			Type t = (Type)registred[message];
			if(t != handler)
				throw new Exception("Type " + handler.Name + " not registred for message " + message);
#endif
			registred.Remove(message);
			Trace.WriteLine(handler.Name + " UNregistred for message " + message);
		}

		private void ListenerStart()
		{
			TcpListener serv = null;
			try
			{
				IPAddress ipAddress = MyDNS.ResolveIP(Globals.Instance.LocalCopyName);
				serv = new TcpListener(ipAddress,ListenPort);
				serv.Start();
				Trace.WriteLine("Listerner started on port:"+ ListenPort);
				while(true)
				{
					if(!serv.Pending())
					{
						Thread.Sleep(PollingTime);
						continue;
					}
					Socket skt = serv.AcceptSocket();
					NetworkStream sktstr = new NetworkStream(skt);
					string req = Bertaccini.Utils.SRead.ReadLine(sktstr);// sr.ReadLine();
					Trace.WriteLine("Listener: received "+ req);
					object oh = registred[req];
					if(oh != null)
					{
						Type t = (Type)oh;
						IMessageHandler ml = (IMessageHandler)Activator.CreateInstance(t,new object[]{req,skt,sktstr});
						ml.Go();
					}
					else
					{
						skt.Shutdown(SocketShutdown.Both);
						skt.Close();
						Trace.WriteLine("No handler registred for this message");
					}
				}
			}
			catch(ThreadAbortException)
			{
				serv.Stop();
			}
		}
		#region IDisposable Members
		/// <summary>
		/// Rilascia le risorse utilizzate dall'istanza, terminando il thread
		/// di ascolto.
		/// </summary>
		public void Dispose()
		{
			t.Abort();
		}

		#endregion
	}
}
